<?php
require __DIR__ . '/../config.php';
set_cors();
if (is_options()) exit(0);

/**
 * Resolve a short share id into a canonical, validated payload.
 *
 * Request:
 *   GET /api/share/resolve?id={shortId}[&format=json]
 *
 * Responses:
 *   200 OK:
 *     {
 *       "v": 1,
 *       "id": "JqAveU3N",
 *       "status": "active",
 *       "type": "atw.share",         // optional
 *       "title": "Ask The Word …",   // optional
 *       "payload": { ... },          // canonical, server-validated
 *       "createdAt": "2025-10-31T18:22:11Z",
 *       "expiresAt": "2025-12-01T00:00:00Z",   // or null
 *       "og": { "title": "...", "desc": "...", "image": "https://..." },
 *       "urls": {
 *         "share": "https://go.snickitybit.com/share/JqAveU3N",
 *         "ul":    "https://go.snickitybit.com/ul/JqAveU3N",
 *         "long":  "https://snickitybit.com/h?...sig=..."   // only if HMAC_SECRET is defined; otherwise omitted
 *       }
 *     }
 *
 *   410 Gone:
 *     { "status": "expired", "id": "...", "urls": { "share": "...", "ul": "..." } }
 *
 *   403 Forbidden:
 *     { "status": "revoked", "id": "...", "urls": { "share": "...", "ul": "..." } }
 *
 *   404 Not Found:
 *     { "error": "Not found" }
 *
 * Behavior:
 *   - If ?format=json -> JSON body (recommended for apps).
 *   - Else (default)  -> 302 redirect to /share/{id} so links work in browsers.
 */

header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
header('Pragma: no-cache');
header('X-Content-Type-Options: nosniff');
header('Referrer-Policy: no-referrer');

$method   = $_SERVER['REQUEST_METHOD'] ?? 'GET';
$wantJson = (($_GET['format'] ?? '') === 'json');

// Only allow GET/HEAD
if (!in_array($method, ['GET', 'HEAD'], true)) {
  if ($wantJson) send_json(405, ['error' => 'Method not allowed']);
  http_response_code(405);
  exit;
}

// Inputs
$id = $_GET['id'] ?? '';
if (!is_valid_id($id)) {
  if ($wantJson) send_json(400, ['error' => 'Invalid id']);
  http_response_code(400);
  exit;
}

// URL bases (adjust if your domains change)
$DOMAIN_GO = 'https://go.snickitybit.com';
$SHARE_URL = $DOMAIN_GO . '/share/' . $id; // browser landing page (OG tags, "Open in App" button)
$UL_URL    = $DOMAIN_GO . '/ul/' . $id;    // universal-link path your iOS app claims

// Optional long URL (signed) support -----------------------------------------
$longUrl = null;
if (defined('HMAC_SECRET') && HMAC_SECRET) {
  $hmacKey = base64_decode(HMAC_SECRET, true);
  if ($hmacKey !== false && strlen($hmacKey) > 0) {
    // We'll compute it later if/when we fetch payload_json
  } else {
    $hmacKey = null; // treat as not configured
  }
} else {
  $hmacKey = null;
}
function b64url($raw) {
  return rtrim(strtr(base64_encode($raw), '+/', '-_'), '=');
}

// Query database --------------------------------------------------------------
$db = pdo();
$stmt = $db->prepare(
  'SELECT id, v, type, title, payload_json, status, created_at, expires_at,
          og_title, og_desc, og_image_url,
          (expires_at IS NULL OR expires_at > UTC_TIMESTAMP()) AS not_expired
   FROM links
   WHERE id = ?
   LIMIT 1'
);
$stmt->execute([$id]);
$row = $stmt->fetch(PDO::FETCH_ASSOC);

if (!$row) {
  if ($wantJson) send_json(404, ['error' => 'Not found']);
  // Non-JSON: fall back to 404 with no body
  http_response_code(404);
  exit;
}

// Basics
$status     = $row['status']; // 'active' | 'revoked' | etc.
$notExpired = (int)($row['not_expired'] ?? 0) === 1;

$createdAt  = $row['created_at'] ?: null;
$expiresAt  = $row['expires_at'] ?: null;

// Construct optional longUrl if HMAC is configured
$payloadJson = (string)($row['payload_json'] ?? '{}');
if ($hmacKey) {
  $sigRaw  = hash_hmac('sha256', $payloadJson, $hmacKey, true);
  $base    = 'https://snickitybit.com/h';
  $longUrl = $base . '?v=1&d=' . b64url($payloadJson) . '&sig=' . b64url($sigRaw);
}

// Common URL bundle
$urls = [
  'share' => $SHARE_URL,
  'ul'    => $UL_URL,
];
if ($longUrl) {
  $urls['long'] = $longUrl;
}

// Status handling -------------------------------------------------------------
if ($status !== 'active') {
  // Revoked → 403
  if ($status === 'revoked') {
    $body = ['status' => 'revoked', 'id' => $id, 'urls' => $urls];
    if ($wantJson) send_json(403, $body);
    http_response_code(403);
    exit;
  }
  // Any other non-active state → 404
  if ($wantJson) send_json(404, ['error' => 'Not found']);
  http_response_code(404);
  exit;
}

if (!$notExpired) {
  // Expired → 410
  $body = ['status' => 'expired', 'id' => $id, 'urls' => $urls];
  if ($wantJson) send_json(410, $body);
  http_response_code(410);
  exit;
}

// We’re active: increment counters (best effort; ignore failures)
try {
  $upd = $db->prepare('UPDATE links SET clicks = clicks + 1, last_access = UTC_TIMESTAMP() WHERE id = ?');
  $upd->execute([$id]);
} catch (Throwable $e) {
  // no-op
}

// Decode payload safely
$payload = json_decode($payloadJson, true);
if (!is_array($payload)) $payload = new stdClass();

// Build JSON result
$result = [
  'v'         => (int)($row['v'] ?? 1),
  'id'        => $row['id'],
  'status'    => 'active',
  'type'      => $row['type'] ?: null,
  'title'     => $row['title'] ?: null,
  'payload'   => $payload,
  'createdAt' => $createdAt,
  'expiresAt' => $expiresAt,
  'og'        => [
    'title' => $row['og_title'] ?: null,
    'desc'  => $row['og_desc'] ?: null,
    'image' => $row['og_image_url'] ?: null,
  ],
  'urls'      => $urls,
];

// Output ----------------------------------------------------------------------
if ($wantJson) {
  // JSON flow for apps / programmatic use
  send_json(200, $result);
  exit;
}

// Non-JSON flow: redirect to browser-friendly landing page (/share/{id})
// Keep it simple and always 302 to the display page.
header('Location: ' . $SHARE_URL, true, 302);
exit;
