<?php
// Public landing page for SEO/social with OG tags
// Location: /home/.../public_html/go.snickitybit.com/share.php

ini_set('display_errors', 1);
error_reporting(E_ALL);

require __DIR__ . '/config.php'; // must provide pdo(), is_valid_id(), etc.

try {
    $id = $_GET['id'] ?? '';
    if (!function_exists('is_valid_id') || !is_valid_id($id)) {
        http_response_code(400);
        header('Content-Type: text/plain; charset=UTF-8');
        echo "Bad Request: invalid id";
        exit;
    }

    $db = pdo();

    // --- helpers --- //
    $colExists = function(string $table, string $col) use ($db): bool {
        $q = $db->prepare("SELECT COUNT(*)
                             FROM information_schema.COLUMNS
                            WHERE table_schema = DATABASE()
                              AND table_name = :t
                              AND column_name = :c");
        $q->execute([':t'=>$table, ':c'=>$col]);
        return ((int)$q->fetchColumn()) > 0;
    };

    // Column presence (schema guards)
    $hasStatus        = $colExists('links','status');
    $hasExpiresAt     = $colExists('links','expires_at');
    $hasCanonical     = $colExists('links','canonical_url');
    $hasLongUrl       = $colExists('links','long_url');
    $hasTitle         = $colExists('links','title');
    $hasDescription   = $colExists('links','description');
    $hasOgTitle       = $colExists('links','og_title');
    $hasOgDesc        = $colExists('links','og_desc');
    $hasOgImageUrl    = $colExists('links','og_image_url');
    $hasClicks        = $colExists('links','clicks');
    $hasLastAccess    = $colExists('links','last_access');
    $hasSharedCount   = $colExists('links','shared_count');
    $hasLastSharedAt  = $colExists('links','last_shared_at');
    if ($colExists('links','created_at')) { /* no-op; presence checked to silence warnings */ }

    // Build SELECT list based on what exists
    $cols = ['id'];
    if ($hasStatus)        $cols[] = 'status';
    if ($hasExpiresAt)     $cols[] = 'expires_at';
    if ($hasCanonical)     $cols[] = 'canonical_url';
    if ($hasLongUrl)       $cols[] = 'long_url';
    if ($hasTitle)         $cols[] = 'title';
    if ($hasDescription)   $cols[] = 'description';
    if ($hasOgTitle)       $cols[] = 'og_title';
    if ($hasOgDesc)        $cols[] = 'og_desc';
    if ($hasOgImageUrl)    $cols[] = 'og_image_url';
    if ($hasClicks)        $cols[] = 'clicks';
    if ($hasLastAccess)    $cols[] = 'last_access';
    if ($hasSharedCount)   $cols[] = 'shared_count';
    if ($hasLastSharedAt)  $cols[] = 'last_shared_at';

    $select = implode(', ', $cols);
    $stmt = $db->prepare("SELECT $select FROM links WHERE id = ? LIMIT 1");
    $stmt->execute([$id]);
    $row = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$row) {
        http_response_code(404);
        header('Content-Type: text/plain; charset=UTF-8');
        echo "Not Found";
        exit;
    }

    // ------- state ------- //
    $status   = strtolower($row['status'] ?? 'active');
    $expired  = false;
    if ($hasExpiresAt && !empty($row['expires_at'])) {
        // assume UTC in DB
        $expired = (strtotime($row['expires_at'] . ' UTC') <= time());
    }

    if ($status === 'revoked') {
        http_response_code(403);
    } elseif ($expired || $status === 'expired') {
        http_response_code(410);
    } else {
        http_response_code(200);
    }

    // ------- URLs & metadata (go-only) ------- //
    $base      = rtrim(SHORT_DOMAIN, '/');                  // e.g., https://go.snickitybit.com
    $pageUrl   = $base . '/share/' . urlencode($id);        // This landing page
    $dynamicOg = $base . '/share/og/' . urlencode($id) . '.png';
    $ulUrl     = $base . '/s/' . urlencode($id);            // Deep link (Universal Link)
    $webButtonUrl = $base . '/s/index.php?id=' . urlencode($id); // Basic web viewer

    // Prefer table values for metadata only
    $canonicalUrl = $hasCanonical && !empty($row['canonical_url']) ? $row['canonical_url'] : null;
    $longUrlDb    = $hasLongUrl   && !empty($row['long_url'])      ? $row['long_url']      : null;

    // Title/description (prefer og_* if present, else fallback)
    $ogTitle = '';
    $ogDesc  = '';
    if ($hasOgTitle) { $ogTitle = trim((string)($row['og_title'] ?? '')); }
    if ($hasOgDesc)  { $ogDesc  = trim((string)($row['og_desc']  ?? '')); }
    if ($ogTitle === '' && $hasTitle)        $ogTitle = trim((string)($row['title'] ?? ''));
    if ($ogDesc  === ''  && $hasDescription) $ogDesc  = trim((string)($row['description'] ?? ''));

    if ($ogTitle === '') $ogTitle = "Ask His Word — Shared Insight";
    if ($ogDesc  === '') $ogDesc  = "Open this shared Bible insight in Ask His Word.";

    // OG image: prefer table value if present, else dynamic generator
    $ogImageUrl = ($hasOgImageUrl && !empty($row['og_image_url']))
                ? $row['og_image_url']
                : $dynamicOg;

    // Stats (optional)
    $sharedCount  = ($hasSharedCount   && isset($row['shared_count']))  ? (int)$row['shared_count'] : null;
    $lastSharedAt = ($hasLastSharedAt  && !empty($row['last_shared_at'])) ? $row['last_shared_at']  : null;
    $clicks       = ($hasClicks        && isset($row['clicks']))        ? (int)$row['clicks']       : null;
    $lastAccess   = ($hasLastAccess    && !empty($row['last_access']))  ? $row['last_access']       : null;

    header('Content-Type: text/html; charset=UTF-8');
    header('Cache-Control: public, max-age=300');

} catch (Throwable $e) {
    http_response_code(500);
    header('Content-Type: text/plain; charset=UTF-8');
    echo "Internal Error: " . $e->getMessage();
    exit;
}

?>
<!DOCTYPE html>
<html lang="en">
<head>
<meta charset="utf-8">
<title><?= htmlspecialchars($ogTitle) ?></title>
<meta name="viewport" content="width=device-width, initial-scale=1">
<meta name="description" content="<?= htmlspecialchars($ogDesc) ?>">

<link rel="canonical" href="<?= htmlspecialchars($pageUrl) ?>">

<!-- Open Graph -->
<meta property="og:type" content="article">
<meta property="og:title" content="<?= htmlspecialchars($ogTitle) ?>">
<meta property="og:description" content="<?= htmlspecialchars($ogDesc) ?>">
<meta property="og:url" content="<?= htmlspecialchars($pageUrl) ?>">
<meta property="og:image" content="<?= htmlspecialchars($ogImageUrl) ?>">
<meta property="og:site_name" content="Ask His Word">
<meta property="og:locale" content="en_US">
<meta property="og:image:width" content="1200">
<meta property="og:image:height" content="630">

<!-- Twitter Card -->
<meta name="twitter:card" content="summary_large_image">
<meta name="twitter:title" content="<?= htmlspecialchars($ogTitle) ?>">
<meta name="twitter:description" content="<?= htmlspecialchars($ogDesc) ?>">
<meta name="twitter:image" content="<?= htmlspecialchars($ogImageUrl) ?>">

<style>
:root { color-scheme: light dark; }
body { margin:0; font-family: system-ui, -apple-system, Segoe UI, Roboto, Ubuntu, Cantarell, "Helvetica Neue", Arial;
       background: radial-gradient(1200px 600px at 70% 0%, rgba(0,0,0,0.06), transparent), #f6f8ff; }
.container { max-width: 720px; margin: 10vh auto; padding: 24px; }
.card { background: white; border-radius: 16px; padding: 20px;
        box-shadow: 0 10px 30px rgba(0,0,0,0.08); border: 1px solid rgba(0,0,0,0.05); }
h1 { font-size: 22px; margin: 0 0 6px; }
p  { color: #555; line-height: 1.5; margin: 8px 0 0; }
.actions { margin-top: 16px; display: flex; gap: 10px; flex-wrap: wrap; }
a.button { background: #2563eb; color: white; padding: 10px 14px; border-radius: 10px; text-decoration: none; font-weight: 600; }
.thumb { width: 100%; aspect-ratio: 1000/525; border-radius: 12px; object-fit: cover; margin: 10px 0 14px; background: #dbeafe; }
.badge { display:inline-block; background:#eef2ff; color:#3730a3; padding:6px 10px; border-radius:999px; font-size:12px; font-weight:600; }
.status { margin: 10px 0 0; font-size: 13px; color: #666; }
.meta  { margin-top: 12px; font-size: 13px; color:#444; display:flex; gap:14px; flex-wrap:wrap; }
.meta .chip { background:#f1f5f9; border:1px solid #e2e8f0; padding:6px 10px; border-radius:999px; }
</style>
</head>
<body>
  <div class="container">
    <div class="card">
      <span class="badge">Ask His Word</span>
      <h1><?= htmlspecialchars($ogTitle) ?></h1>
      <img src="<?= htmlspecialchars($ogImageUrl) ?>" alt="Preview image" class="thumb" loading="lazy">
      <p><?= htmlspecialchars($ogDesc) ?></p>

      <?php if ($sharedCount !== null || $clicks !== null || $lastSharedAt || $lastAccess): ?>
        <div class="meta">
          <?php if ($sharedCount !== null): ?>
            <span class="chip">
              <?= htmlspecialchars($sharedCount) ?> <?= ($sharedCount == 1 ? 'share' : 'shares') ?>
            </span>
          <?php endif; ?>
          <?php if ($clicks !== null): ?>
            <span class="chip">
              <?= htmlspecialchars($clicks) ?> <?= ($clicks == 1 ? 'view' : 'views') ?>
            </span>
          <?php endif; ?>
          <?php if ($lastSharedAt): ?>
            <span class="chip">Last shared: <?= htmlspecialchars($lastSharedAt) ?> UTC</span>
          <?php endif; ?>
          <?php if ($lastAccess): ?>
            <span class="chip">Last viewed: <?= htmlspecialchars($lastAccess) ?> UTC</span>
          <?php endif; ?>
        </div>
      <?php endif; ?>

      <div class="actions">
        <!-- /s/* is claimed by AASA; this opens the app when installed -->
        <a class="button" href="<?= htmlspecialchars($ulUrl) ?>">Open in App</a>
        <!-- Use your existing web viewer -->
        <a class="button" href="<?= htmlspecialchars($webButtonUrl) ?>">Open on Web</a>
      </div>

      <?php if (($row['status'] ?? 'active') !== 'active' || (isset($expired) && $expired)): ?>
        <p class="status">
          <?php
            $statusTxt = strtolower($row['status'] ?? 'active');
            if ($statusTxt === 'revoked')      echo "This link has been revoked.";
            elseif (!empty($expired) || $statusTxt === 'expired') echo "This link has expired.";
            elseif ($statusTxt === 'purged')   echo "This link has been removed.";
          ?>
        </p>
      <?php endif; ?>
    </div>
  </div>
</body>
</html>
<?php
function absolute_url(string $path): string {
    $base = rtrim(SHORT_DOMAIN, '/');
    if ($path && $path[0] !== '/') $path = '/' . $path;
    return $base . $path;
}
