<?php
// ✅ stats.php lives in /api; config.php is one level up.
require __DIR__ . '/../config.php';

// DEV: show PHP errors (remove or disable in production)
ini_set('display_errors', 1);
error_reporting(E_ALL);

set_cors();
if (is_options()) exit(0);

/* ----------------------- Helpers (safe outside try) ----------------------- */

function parse_date($s) {
    if (!$s) return null;
    // Expect YYYY-MM-DD
    if (!preg_match('/^\d{4}-\d{2}-\d{2}$/', $s)) return null;
    return $s;
}

function env_has_link_events(PDO $db) {
    try {
        $sql = "SELECT COUNT(*) FROM information_schema.tables 
                WHERE table_schema = DATABASE() AND table_name = 'link_events'";
        return (int)$db->query($sql)->fetchColumn() > 0;
    } catch (Throwable $e) {
        return false; // assume no events table if we can't probe
    }
}

/** Check if a column exists on a table. */
function col_exists(PDO $db, string $table, string $column): bool {
    try {
        $stmt = $db->prepare(
            "SELECT COUNT(*) 
               FROM information_schema.COLUMNS 
              WHERE table_schema = DATABASE() 
                AND table_name = :t 
                AND column_name = :c"
        );
        $stmt->execute([':t'=>$table, ':c'=>$column]);
        return ((int)$stmt->fetchColumn()) > 0;
    } catch (Throwable $e) {
        return false;
    }
}

/** Inclusive UTC range: produces "<col> BETWEEN :from_utc AND :to_utc" */
function rangeWhere(string $col): string {
    return "$col BETWEEN :from_utc AND :to_utc";
}

/**
 * Build the bucket expression for GROUP BY in the requested TZ,
 * falling back to UTC (the raw column) if MySQL TZ tables aren't loaded.
 */
function groupExpr(string $column, string $group): string {
    if ($group === 'day') {
        return "DATE(COALESCE(CONVERT_TZ($column, '+00:00', :tz), $column))";
    } else {
        return "DATE_FORMAT(COALESCE(CONVERT_TZ($column, '+00:00', :tz), $column), '%Y-%m-%d %H:00:00')";
    }
}

/* --------------------------------- Main ---------------------------------- */

try {
    // Admin auth via SECRET_API_KEY (same as revoke/purge)
    require_api_key(); // expects header X-API-Key: <SECRET_API_KEY>

    // --- normalize params ---
    $in_from = parse_date($_GET['from'] ?? '');
    $in_to   = parse_date($_GET['to']   ?? '');
    $tz      = $_GET['tz']   ?? 'UTC';
    $group   = $_GET['group'] ?? 'day';   // 'day' | 'hour'
    $mode    = $_GET['mode']  ?? 'both';  // 'summary' | 'timeseries' | 'both'

    // date range defaults: last 30 days inclusive (to = today)
    $todayUTC = new DateTime('now', new DateTimeZone('UTC'));
    $default_to   = $todayUTC->format('Y-m-d');
    $default_from = $todayUTC->sub(new DateInterval('P29D'))->format('Y-m-d');

    $from = $in_from ?: $default_from;
    $to   = $in_to   ?: $default_to;

    // validate tz (best effort). If invalid, fallback to UTC.
    try { new DateTimeZone($tz); } catch (Throwable $e) { $tz = 'UTC'; }

    // ensure group/mode valid
    if ($group !== 'day' && $group !== 'hour') $group = 'day';
    if (!in_array($mode, ['summary','timeseries','both'], true)) $mode = 'both';

    // compute inclusive range [from 00:00:00, to 23:59:59] in TZ; convert to UTC for DB filtering
    $from_dt_tz = new DateTime($from . ' 00:00:00', new DateTimeZone($tz));
    $to_dt_tz   = new DateTime($to   . ' 23:59:59', new DateTimeZone($tz));
    $from_dt_utc = clone $from_dt_tz; $from_dt_utc->setTimezone(new DateTimeZone('UTC'));
    $to_dt_utc   = clone $to_dt_tz;   $to_dt_utc->setTimezone(new DateTimeZone('UTC'));

    $db = pdo();

    // Environment capabilities
    $hasEvents        = env_has_link_events($db);
    $hasRevokedAt     = col_exists($db, 'links', 'revoked_at'); // probably false
    $hasExpiresAt     = col_exists($db, 'links', 'expires_at');
    $hasClicks        = col_exists($db, 'links', 'clicks');
    $hasLastAccess    = col_exists($db, 'links', 'last_access');
    $hasSharedCount   = col_exists($db, 'links', 'shared_count');
    $hasLastSharedAt  = col_exists($db, 'links', 'last_shared_at');

    $capabilities = [
        'uses_events'        => $hasEvents,
        'uses_revoked_at'    => $hasRevokedAt,
        'uses_expires_at'    => $hasExpiresAt,
        'has_clicks'         => $hasClicks,
        'has_last_access'    => $hasLastAccess,
        'has_shared_count'   => $hasSharedCount,
        'has_last_shared_at' => $hasLastSharedAt,
        // If no events table, "shares over time" are approximated by last_shared_at
        'shares_are_approx'  => !$hasEvents,
        'revoked_is_approx'  => !$hasRevokedAt,
    ];

    /* ---------------------------- SUMMARY QUERIES ---------------------------- */

    $summary = null;
    if ($mode === 'summary' || $mode === 'both') {
        $bindSummary = [
            ':from_utc' => $from_dt_utc->format('Y-m-d H:i:s'),
            ':to_utc'   => $to_dt_utc->format('Y-m-d H:i:s'),
        ];

        // created in range
        $sql_created = "SELECT COUNT(*) FROM links WHERE " . rangeWhere('created_at');

        // revoked in range
        if ($hasRevokedAt) {
            $sql_revoked = "SELECT COUNT(*) FROM links
                             WHERE status='revoked' AND " . rangeWhere('revoked_at');
        } else {
            $sql_revoked = "SELECT COUNT(*) FROM links
                             WHERE status='revoked' AND " . rangeWhere('created_at');
        }

        // expired in range
        if ($hasExpiresAt) {
            $sql_expired = "SELECT COUNT(*) FROM links
                             WHERE expires_at IS NOT NULL AND " . rangeWhere('expires_at');
        } else {
            $sql_expired = null;
        }

        // clicks
        if ($hasEvents) {
            $sql_clicks = "SELECT COUNT(*) FROM link_events
                           WHERE event='resolve_ok' AND " . rangeWhere('occurred_at');
        } else if ($hasClicks) {
            // Not precise per-range, but show cumulative clicks for links created in window
            $sql_clicks = "SELECT COALESCE(SUM(clicks),0) FROM links
                           WHERE " . rangeWhere('created_at');
        } else {
            $sql_clicks = null;
        }

        // shares
        // - shares_total: SUM(shared_count) over all rows (global total)
        // - shares_recent: COUNT(*) where last_shared_at falls in range (approx recent activity)
        $shares_total = 0;
        $shares_recent = 0;
        if ($hasSharedCount) {
            $shares_total = (int)$db->query("SELECT COALESCE(SUM(shared_count),0) FROM links")->fetchColumn();
        }
        if ($hasLastSharedAt) {
            $stmt = $db->prepare("SELECT COUNT(*) FROM links WHERE last_shared_at IS NOT NULL AND " . rangeWhere('last_shared_at'));
            $stmt->execute($bindSummary);
            $shares_recent = (int)$stmt->fetchColumn();
        }

        // Run queries
        $stmt = $db->prepare($sql_created);
        $stmt->execute($bindSummary);
        $created_total = (int)$stmt->fetchColumn();

        $stmt = $db->prepare($sql_revoked);
        $stmt->execute($bindSummary);
        $revoked_total = (int)$stmt->fetchColumn();

        if ($sql_expired) {
            $stmt = $db->prepare($sql_expired);
            $stmt->execute($bindSummary);
            $expired_total = (int)$stmt->fetchColumn();
        } else {
            $expired_total = 0;
        }

        if ($sql_clicks) {
            $stmt = $db->prepare($sql_clicks);
            $stmt->execute($bindSummary);
            $clicks_total = (int)$stmt->fetchColumn();
        } else {
            $clicks_total = 0;
        }

        // status breakdown now (global snapshot)
        $status_breakdown = ['active'=>0,'revoked'=>0,'expired'=>0,'purged'=>0];
        foreach ($db->query("SELECT status, COUNT(*) c FROM links GROUP BY status") as $row) {
            $status = strtolower((string)$row['status']);
            $c = (int)$row['c'];
            if (!isset($status_breakdown[$status])) $status_breakdown[$status] = 0;
            $status_breakdown[$status] += $c;
        }

        // active_now = active and not expired as of now (UTC)
        if ($hasExpiresAt) {
            $active_now = (int)$db->query(
                "SELECT COUNT(*) FROM links
                  WHERE status='active'
                    AND (expires_at IS NULL OR expires_at > UTC_TIMESTAMP())"
            )->fetchColumn();
        } else {
            $active_now = (int)$db->query(
                "SELECT COUNT(*) FROM links WHERE status='active'"
            )->fetchColumn();
        }

        $summary = [
            'created'       => $created_total,
            'clicks'        => $clicks_total,
            'revoked'       => $revoked_total,
            'expired'       => $expired_total,
            'active_now'    => $active_now,
            'status'        => $status_breakdown,
            'uses_events'   => $hasEvents,
            'shares_total'  => $shares_total,
            'shares_recent' => $shares_recent,
        ];
    }

    /* --------------------------- TIMESERIES QUERIES -------------------------- */

    $timeseries = null;
    if ($mode === 'timeseries' || $mode === 'both') {
        $tzParam = $tz;

        // created per bucket
        $sql_created_ts =
            "SELECT " . groupExpr('created_at', $group) . " AS bucket, COUNT(*) c
               FROM links
              WHERE " . rangeWhere('created_at') . "
           GROUP BY bucket
           ORDER BY bucket";

        // revoked per bucket
        if ($hasRevokedAt) {
            $sql_revoked_ts =
                "SELECT " . groupExpr('revoked_at', $group) . " AS bucket, COUNT(*) c
                   FROM links
                  WHERE status='revoked'
                    AND " . rangeWhere('revoked_at') . "
               GROUP BY bucket
               ORDER BY bucket";
        } else {
            $sql_revoked_ts =
                "SELECT " . groupExpr('created_at', $group) . " AS bucket, COUNT(*) c
                   FROM links
                  WHERE status='revoked'
                    AND " . rangeWhere('created_at') . "
               GROUP BY bucket
               ORDER BY bucket";
        }

        // expired per bucket
        if ($hasExpiresAt) {
            $sql_expired_ts =
                "SELECT " . groupExpr('expires_at', $group) . " AS bucket, COUNT(*) c
                   FROM links
                  WHERE expires_at IS NOT NULL
                    AND " . rangeWhere('expires_at') . "
               GROUP BY bucket
               ORDER BY bucket";
        } else {
            $sql_expired_ts = null;
        }

        // clicks per bucket
        if ($hasEvents) {
            $sql_clicks_ts =
                "SELECT " . groupExpr('occurred_at', $group) . " AS bucket, COUNT(*) c
                   FROM link_events
                  WHERE event='resolve_ok'
                    AND " . rangeWhere('occurred_at') . "
               GROUP BY bucket
               ORDER BY bucket";
        } else if ($hasClicks) {
            $sql_clicks_ts =
                "SELECT " . groupExpr('created_at', $group) . " AS bucket, COALESCE(SUM(clicks),0) c
                   FROM links
                  WHERE " . rangeWhere('created_at') . "
               GROUP BY bucket
               ORDER BY bucket";
        } else {
            $sql_clicks_ts = null;
        }

        // shared per bucket (approx via last_shared_at)
        $sql_shared_ts = null;
        if ($hasLastSharedAt) {
            $sql_shared_ts =
                "SELECT " . groupExpr('last_shared_at', $group) . " AS bucket, COUNT(*) c
                   FROM links
                  WHERE last_shared_at IS NOT NULL
                    AND " . rangeWhere('last_shared_at') . "
               GROUP BY bucket
               ORDER BY bucket";
        }

        $bind = [
            ':from_utc' => $from_dt_utc->format('Y-m-d H:i:s'),
            ':to_utc'   => $to_dt_utc->format('Y-m-d H:i:s'),
            ':tz'       => $tzParam,
        ];

        $fetch_map = function(?string $sql) use ($db, $bind) {
            if (!$sql) return [];
            $stmt = $db->prepare($sql);
            $stmt->execute($bind);
            $out = [];
            while ($r = $stmt->fetch(PDO::FETCH_ASSOC)) {
                $out[$r['bucket']] = (int)$r['c'];
            }
            return $out;
        };

        $created_map = $fetch_map($sql_created_ts);
        $revoked_map = $fetch_map($sql_revoked_ts);
        $expired_map = $fetch_map($sql_expired_ts);
        $clicks_map  = $fetch_map($sql_clicks_ts);
        $shared_map  = $fetch_map($sql_shared_ts);

        // Dense series aligned to requested TZ/group
        $keys_tz = [];
        $cursor = new DateTime($from . ' 00:00:00', new DateTimeZone($tz));
        $end    = new DateTime($to   . ' 23:59:59', new DateTimeZone($tz));
        while ($cursor <= $end) {
            $keys_tz[] = ($group === 'day')
                ? $cursor->format('Y-m-d')
                : $cursor->format('Y-m-d H:00:00');
            $cursor->modify($group === 'day' ? '+1 day' : '+1 hour');
        }

        $rows = [];
        foreach ($keys_tz as $k) {
            $rows[] = [
                'bucket'  => $k,
                'created' => $created_map[$k] ?? 0,
                'clicks'  => $clicks_map[$k]  ?? 0,
                'revoked' => $revoked_map[$k] ?? 0,
                'expired' => $expired_map[$k] ?? 0,
                'shared'  => $shared_map[$k]  ?? 0, // approx
            ];
        }

        $timeseries = $rows;
    }

    /* -------------------------------- Response ------------------------------- */

    send_json(200, [
        'range' => [
            'from'  => $from,
            'to'    => $to,
            'tz'    => $tz,
            'group' => $group,
        ],
        'mode'         => $mode,
        'summary'      => $summary,
        'timeseries'   => $timeseries,
        'capabilities' => $capabilities,
    ], [
        'Cache-Control' => 'no-store',
    ]);

} catch (Throwable $e) {
    // DEV-friendly error JSON instead of a blank 500
    send_json(500, [
        'error'  => 'internal_error',
        'detail' => $e->getMessage(),   // remove in prod if you want
    ], ['Cache-Control' => 'no-store']);
}
