<?php
require __DIR__ . '/../config.php';
set_cors();
if (is_options()) exit(0);
require_api_key(); // Admin-only

if (($_SERVER['REQUEST_METHOD'] ?? '') !== 'POST') {
    send_json(405, ['error' => 'Method not allowed']);
}

// Accept either:
//  - query:  ?id=abcd1234&expireNow=1
//  - body:   { "id": "abcd1234", "expireNow": true }
//  - body:   { "ids": ["id1","id2"], "expireNow": false }
$in   = json_input();
$idQ  = $_GET['id'] ?? '';
$idsB = isset($in['ids']) && is_array($in['ids']) ? $in['ids'] : null;
$idB  = isset($in['id'])  && is_string($in['id'])  ? $in['id']  : '';
$expireNow =
    (isset($_GET['expireNow']) && ($_GET['expireNow'] === '1' || $_GET['expireNow'] === 'true')) ||
    (isset($in['expireNow']) && ($in['expireNow'] === true || $in['expireNow'] === 1));

$targets = [];
if ($idsB) {
    $targets = array_values(array_filter($idsB, fn($s) => is_string($s) && $s !== ''));
} elseif ($idB !== '') {
    $targets = [$idB];
} elseif ($idQ !== '') {
    $targets = [$idQ];
}

if (empty($targets)) {
    send_json(400, ['error' => 'Missing id or ids']);
}

// Validate ids
$valid = [];
foreach ($targets as $t) {
    if (is_valid_id($t)) $valid[] = $t;
}
if (empty($valid)) {
    send_json(400, ['error' => 'No valid ids']);
}

$db = pdo();

// We’ll collect per-id results
$results = [];

if ($expireNow) {
    // Force immediate expiry for any matching link regardless of status.
    // This sets expires_at = NOW() but keeps the status as-is (could be active/revoked).
    $stmt = $db->prepare('UPDATE links SET expires_at = UTC_TIMESTAMP() WHERE id = ?');
    foreach ($valid as $id) {
        try {
            $stmt->execute([$id]);
            $results[] = ['id' => $id, 'action' => 'expireNow', 'ok' => true];
        } catch (Throwable $e) {
            $results[] = ['id' => $id, 'action' => 'expireNow', 'ok' => false, 'error' => $e->getMessage()];
        }
    }
} else {
    // Mark as revoked. If already revoked or expired, we still return ok for idempotency.
    $get = $db->prepare('SELECT status, (expires_at IS NULL OR expires_at > UTC_TIMESTAMP()) AS not_expired FROM links WHERE id = ? LIMIT 1');
    $revoke = $db->prepare('UPDATE links SET status = "revoked" WHERE id = ? AND status = "active"');
    foreach ($valid as $id) {
        try {
            $get->execute([$id]);
            $row = $get->fetch();
            if (!$row) {
                // Unknown id -> keep old behavior if you prefer, but nicer UX is to return ok=false
                $results[] = ['id' => $id, 'action' => 'revoke', 'ok' => false, 'error' => 'Not found'];
                continue;
            }

            $status = $row['status'];           // active | revoked | (maybe other)
            $notExpired = (int)$row['not_expired'] === 1;

            if ($status === 'revoked') {
                // Already revoked — idempotent success
                $results[] = ['id' => $id, 'action' => 'revoke', 'ok' => true, 'status' => 'revoked'];
                continue;
            }

            if (!$notExpired) {
                // Already expired — treat as ok, since it can’t be used
                $results[] = ['id' => $id, 'action' => 'revoke', 'ok' => true, 'status' => 'expired'];
                continue;
            }

            // Active & not expired -> revoke now
            $revoke->execute([$id]);
            $results[] = ['id' => $id, 'action' => 'revoke', 'ok' => true, 'status' => 'revoked'];
        } catch (Throwable $e) {
            $results[] = ['id' => $id, 'action' => 'revoke', 'ok' => false, 'error' => $e->getMessage()];
        }
    }
}

// Overall ok if at least one succeeded
$anyOk = array_reduce($results, fn($carry, $r) => $carry || (!empty($r['ok'])), false);
$code = $anyOk ? 200 : 400;

send_json($code, [
    'ok' => $anyOk,
    'count' => count($results),
    'results' => $results
]);
