<?php
require __DIR__ . '/../config.php';
set_cors();
if (is_options()) exit(0);

/**
 * Resolve a short share id into a canonical, validated payload.
 *
 * Request:
 *   GET /api/share/resolve?id={shortId}[&format=json]
 *
 * Responses:
 *   200 OK:
 *     {
 *       "v": 1,
 *       "id": "JqAveU3N",
 *       "status": "active",
 *       "type": "atw.share",         // optional
 *       "title": "Ask His Word …",   // optional
 *       "payload": { ... },          // canonical, server-validated
 *       "createdAt": "2025-10-31T18:22:11Z",
 *       "expiresAt": "2025-12-01T00:00:00Z",   // or null
 *       "og": { "title": "...", "desc": "...", "image": "https://..." },
 *       "urls": {
 *         "share": "https://go.snickitybit.com/share/JqAveU3N",
 *         "ul":    "https://go.snickitybit.com/s/JqAveU3N",
 *         "long":  "https://snickitybit.com/h?...sig=..."   // only if HMAC_SECRET is defined; otherwise omitted
 *       },
 *       "stats": {
 *         "sharedCount": 12,
 *         "clicks": 34,
 *         "lastSharedAt": "2025-11-03T21:44:09Z",
 *         "lastAccess": "2025-11-04T01:11:00Z"
 *       }
 *     }
 *
 *   410 Gone:
 *     { "status": "expired", "id": "...", "urls": { "share": "...", "ul": "..." } }
 *
 *   403 Forbidden:
 *     { "status": "revoked", "id": "...", "urls": { "share": "...", "ul": "..." } }
 *
 *   404 Not Found:
 *     { "error": "Not found" }
 *
 * Behavior:
 *   - If ?format=json -> JSON body (recommended for apps).
 *   - Else (default)  -> 302 redirect to /share/{id} so links work in browsers.
 */

header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
header('Pragma: no-cache');
header('X-Content-Type-Options: nosniff');
header('Referrer-Policy: no-referrer');

$method   = $_SERVER['REQUEST_METHOD'] ?? 'GET';
$wantJson = (($_GET['format'] ?? '') === 'json');

if (!in_array($method, ['GET', 'HEAD'], true)) {
  if ($wantJson) send_json(405, ['error' => 'Method not allowed']);
  http_response_code(405);
  exit;
}

$id = $_GET['id'] ?? '';
if (!is_valid_id($id)) {
  if ($wantJson) send_json(400, ['error' => 'Invalid id']);
  http_response_code(400);
  exit;
}

// URL bases
$DOMAIN_GO = rtrim(SHORT_DOMAIN, '/');
$SHARE_URL = $DOMAIN_GO . '/share/' . $id; // browser landing page
$UL_URL    = $DOMAIN_GO . '/s/' . $id;     // universal link path

// Optional long URL (signed)
$longUrl = null;
if (defined('HMAC_SECRET') && HMAC_SECRET) {
  $hmacKey = base64_decode(HMAC_SECRET, true);
  if ($hmacKey !== false && strlen($hmacKey) > 0) {
    // ok
  } else {
    $hmacKey = null;
  }
} else {
  $hmacKey = null;
}
function b64url($raw) { return rtrim(strtr(base64_encode($raw), '+/', '-_'), '='); }

$db = pdo();
$stmt = $db->prepare(
  'SELECT id, v, type, title, payload_json, status, created_at, expires_at,
          og_title, og_desc, og_image_url,
          clicks, last_access, shared_count, last_shared_at,
          (expires_at IS NULL OR expires_at > UTC_TIMESTAMP()) AS not_expired
     FROM links
    WHERE id = ?
    LIMIT 1'
);
$stmt->execute([$id]);
$row = $stmt->fetch(PDO::FETCH_ASSOC);

if (!$row) {
  if ($wantJson) send_json(404, ['error' => 'Not found']);
  http_response_code(404);
  exit;
}

$status     = $row['status'];
$notExpired = (int)($row['not_expired'] ?? 0) === 1;
$createdAt  = $row['created_at'] ?: null;
$expiresAt  = $row['expires_at'] ?: null;

$payloadJson = (string)($row['payload_json'] ?? '{}');
if ($hmacKey) {
  $sigRaw  = hash_hmac('sha256', $payloadJson, $hmacKey, true);
  $base    = 'https://snickitybit.com/h';
  $longUrl = $base . '?v=1&d=' . b64url($payloadJson) . '&sig=' . b64url($sigRaw);
}

$urls = [
  'share' => $SHARE_URL,
  'ul'    => $UL_URL,
];
if ($longUrl) $urls['long'] = $longUrl;

if ($status !== 'active') {
  if ($status === 'revoked') {
    $body = ['status' => 'revoked', 'id' => $id, 'urls' => $urls];
    if ($wantJson) send_json(403, $body);
    http_response_code(403);
    exit;
  }
  if ($wantJson) send_json(404, ['error' => 'Not found']);
  http_response_code(404);
  exit;
}

if (!$notExpired) {
  $body = ['status' => 'expired', 'id' => $id, 'urls' => $urls];
  if ($wantJson) send_json(410, $body);
  http_response_code(410);
  exit;
}

// Best-effort click increment
try {
  $upd = $db->prepare('UPDATE links SET clicks = clicks + 1, last_access = UTC_TIMESTAMP() WHERE id = ?');
  $upd->execute([$id]);
} catch (Throwable $e) { /* ignore */ }

$payload = json_decode($payloadJson, true);
if (!is_array($payload)) $payload = new stdClass();

$result = [
  'v'         => (int)($row['v'] ?? 1),
  'id'        => $row['id'],
  'status'    => 'active',
  'type'      => $row['type'] ?: null,
  'title'     => $row['title'] ?: null,
  'payload'   => $payload,
  'createdAt' => $createdAt,
  'expiresAt' => $expiresAt,
  'og'        => [
    'title' => $row['og_title'] ?: null,
    'desc'  => $row['og_desc'] ?: null,
    'image' => $row['og_image_url'] ?: null,
  ],
  'urls'      => $urls,
  'stats'     => [
    'sharedCount' => isset($row['shared_count']) ? (int)$row['shared_count'] : null,
    'clicks'      => isset($row['clicks']) ? (int)$row['clicks'] : null,
    'lastSharedAt'=> $row['last_shared_at'] ?: null,
    'lastAccess'  => $row['last_access'] ?: null,
  ],
];

if ($wantJson) {
  send_json(200, $result);
  exit;
}

header('Location: ' . $SHARE_URL, true, 302);
exit;
